#ifndef _CC_LOG_H_
#define _CC_LOG_H_

#ifdef CC_LOG

#define SPDLOG_DISABLE_DEFAULT_LOGGER

#include "spdlog/spdlog.h"
#include "spdlog/sinks/stdout_color_sinks.h"
#include "spdlog/sinks/basic_file_sink.h"
#include "spdlog/async.h"

namespace CC{

#ifndef __FIEX__
#define __FIEX__
#endif

#ifdef __FS
#undef __FS
#endif
#define __FS(file, line, fmt) ("[" file ":" #line "] " fmt)

#ifdef _FS
#undef _FS
#endif
#define _FS(file, line, fmt) __FS(file, line, fmt)

#ifdef FS
#undef FS
#endif
#define FS(file, fmt) _FS(file, __LINE__, fmt)

#ifdef LF
#undef LF
#endif
#define LF(fmt) FS(__FIEX__, fmt)
 
namespace spd = spdlog;

class CLog
{
public:

    enum ELevel
    {
        trace = 0,
        debug = 1,
        info = 2,
        warn = 3,
        err = 4,
        critical = 5,
        off = 6
    };

public:

    // the dir of filePath must be exist.
    bool Init(ELevel consoleLevel, ELevel fileLevel, const char* filePath)
    {
        if (!filePath)
        {
            return false;
        }

        spd::level::level_enum cl = (spd::level::level_enum)consoleLevel;
        spd::level::level_enum fl = (spd::level::level_enum)fileLevel;

        try
        {
            spd::set_pattern("[%Y%m%d %H:%M:%S.%e]%^[%L]%v%$");

            if (cl != spd::level::off)
            {
                _console = spd::stdout_color_mt("console");
                if (!_console)
                    return false;
                _console->set_level(cl);
            }

            _file = spdlog::basic_logger_mt<spdlog::async_factory>("async_file_logger", filePath);
            if (!_file)
                return false;
            _file->set_level(fl);

            return true;
        }
        catch (const std::exception& ex)
        {
            std::cerr << "CLog init failed: " << ex.what() << std::endl;
            return false;
        }
    }

public:

    template<typename Arg1, typename... Args>
    inline void Trace(const char *fmt, const Arg1 &arg1, const Args &... args)
    {
        _file->trace(fmt, arg1, args...);
        if (_console) _console->trace(fmt, arg1, args...);
    }

    template<typename Arg1, typename... Args>
    inline void Debug(const char *fmt, const Arg1 &arg1, const Args &... args)
    {
        _file->debug(fmt, arg1, args...);
        if (_console) _console->debug(fmt, arg1, args...);
    }

    template<typename Arg1, typename... Args>
    inline void Info(const char *fmt, const Arg1 &arg1, const Args &... args)
    {
        _file->info(fmt, arg1, args...);
        if (_console) _console->info(fmt, arg1, args...);
    }

    template<typename Arg1, typename... Args>
    inline void Warn(const char *fmt, const Arg1 &arg1, const Args &... args)
    {
        _file->warn(fmt, arg1, args...);
        if (_console) _console->warn(fmt, arg1, args...);
    }

    template<typename Arg1, typename... Args>
    inline void Error(const char *fmt, const Arg1 &arg1, const Args &... args)
    {
        _file->error(fmt, arg1, args...);
        if (_console) _console->error(fmt, arg1, args...);
    }

    template<typename Arg1, typename... Args>
    inline void Critical(const char *fmt, const Arg1 &arg1, const Args &... args)
    {
        _file->critical(fmt, arg1, args...);
        if (_console) _console->critical(fmt, arg1, args...);
    }

    template<typename T>
    inline void Trace(const T &msg)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->trace(msg);
        });
    }

    template<typename T>
    inline void Debug(const T &msg)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->debug(msg);
        });
    }

    template<typename T>
    inline void Info(const T &msg)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->info(msg);
        });
    }

    template<typename T>
    void Warn(const T &msg)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->warn(msg);
        });
    }

    template<typename T>
    inline void Error(const T &msg)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->error(msg);
        });
    }

    template<typename T>
    inline void Critical(const T &msg)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->critical(msg);
        });
    }

public:

    inline void FlushEvery(int seconds)
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            spdlog::flush_every(std::chrono::seconds(seconds));
        });
    }

    // automatically call flush() if message level >= log_level
    inline void FlushOn(ELevel level)
    {
        spd::level::level_enum l = (spd::level::level_enum)level;
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->flush_on(l);
        });
    }

    virtual void Flush()
    {
        spd::apply_all([&](std::shared_ptr<spd::logger> log)
        {
            log->flush();
        });
    }

public:

    CLog() = default;
    virtual ~CLog()
    {
        // Release all spdlog resources, and drop all loggers in the registry.
        // This is optional (only mandatory if using windows + async log).
        spdlog::shutdown();

        _console.reset();
        _file.reset();
    }

private:

    std::shared_ptr<spd::logger> _console = nullptr;
    std::shared_ptr<spd::logger> _file = nullptr;
};

}

#endif

#endif
